(function () {
  "use strict";

  var B = typeof browser !== "undefined" ? browser : chrome;

  // Public OAuth2 bearer token embedded in X's web client JS
  var BEARER =
    "AAAAAAAAAAAAAAAAAAAAANRILgAAAAAAnNwIzUejRCOuH5E6I8xnZz4puTs" +
    "%3D1Zv7ttfk8LF81IUq16cHjhLTvJu4FA33AGWWjCpTnA";

  var scanning = false;
  var stopRequested = false;

  function getCsrfToken() {
    var m = document.cookie.match(/ct0=([^;]+)/);
    return m ? m[1] : null;
  }

  function getPageUsername() {
    var m = window.location.pathname.match(
      /^\/([A-Za-z0-9_]{1,15})\/(?:followers|verified_followers)/
    );
    return m ? m[1] : null;
  }

  function sleep(ms) {
    return new Promise(function (r) { setTimeout(r, ms); });
  }

  // ── API-based fetching (works in background tabs) ────────────────────

  function apiFetch(url) {
    var csrf = getCsrfToken();
    if (!csrf) return Promise.reject(new Error("NO_CSRF"));

    return fetch(url, {
      headers: {
        "authorization": "Bearer " + BEARER,
        "x-csrf-token": csrf,
        "x-twitter-active-user": "yes",
        "x-twitter-auth-type": "OAuth2Session",
      },
      credentials: "include",
    }).then(function (resp) {
      if (resp.status === 429) throw new Error("RATE_LIMITED");
      if (!resp.ok) throw new Error("HTTP_" + resp.status);
      return resp.json();
    });
  }

  async function fetchFollowersViaAPI(screenName, port) {
    var followers = [];
    var cursor = "-1";
    var rateLimitWaits = 0;

    while (cursor !== "0" && !stopRequested) {
      var url =
        "https://x.com/i/api/1.1/followers/list.json" +
        "?screen_name=" + encodeURIComponent(screenName) +
        "&count=200" +
        "&cursor=" + cursor +
        "&skip_status=true" +
        "&include_user_entities=false";

      try {
        var data = await apiFetch(url);
        rateLimitWaits = 0;

        if (data.users) {
          for (var i = 0; i < data.users.length; i++) {
            followers.push(data.users[i].screen_name);
          }
        }

        cursor = data.next_cursor_str || "0";
        port.postMessage({ type: "progress", count: followers.length });
        await sleep(350);
      } catch (e) {
        if (e.message === "RATE_LIMITED" && rateLimitWaits < 3) {
          rateLimitWaits++;
          port.postMessage({
            type: "progress",
            count: followers.length,
            waiting: true,
          });
          // wait ~65s for rate limit window to reset
          await sleep(65000);
        } else {
          throw e;
        }
      }
    }

    return followers;
  }

  // ── DOM scroll fallback (needs tab in foreground) ────────────────────

  function extractVisibleUsernames() {
    var usernames = new Set();
    var cells = document.querySelectorAll('[data-testid="cellInnerDiv"]');

    cells.forEach(function (cell) {
      cell.querySelectorAll("span").forEach(function (span) {
        var text = span.textContent.trim();
        if (/^@[A-Za-z0-9_]{1,15}$/.test(text)) {
          usernames.add(text.substring(1));
        }
      });

      cell.querySelectorAll('a[role="link"]').forEach(function (link) {
        var href = link.getAttribute("href");
        if (href && /^\/[A-Za-z0-9_]{1,15}$/.test(href)) {
          var c = href.substring(1);
          var reserved = [
            "home","explore","search","notifications","messages",
            "settings","i","compose","logout","login","signup",
          ];
          if (reserved.indexOf(c.toLowerCase()) === -1) {
            usernames.add(c);
          }
        }
      });
    });

    return usernames;
  }

  async function fetchFollowersViaDOM(screenName, port) {
    var all = new Set();
    var stableRounds = 0;

    window.scrollTo(0, 0);
    await sleep(1000);

    while (stableRounds < 5 && !stopRequested) {
      var prev = all.size;
      var visible = extractVisibleUsernames();
      visible.delete(screenName);
      visible.forEach(function (u) { all.add(u); });

      stableRounds = all.size === prev ? stableRounds + 1 : 0;

      port.postMessage({ type: "progress", count: all.size });
      window.scrollBy(0, 600);
      await sleep(1200);
    }

    return Array.from(all);
  }

  // ── Main scan orchestrator ───────────────────────────────────────────

  async function scanFollowers(port) {
    if (scanning) {
      port.postMessage({ type: "error", message: "Scan already in progress." });
      return;
    }

    var pageUser = getPageUsername();
    if (!pageUser) {
      port.postMessage({
        type: "error",
        message: "Navigate to x.com/{username}/followers first.",
      });
      return;
    }

    scanning = true;
    stopRequested = false;
    port.postMessage({ type: "started", account: pageUser });

    try {
      var followers = await fetchFollowersViaAPI(pageUser, port);
      port.postMessage({
        type: "done",
        account: pageUser,
        followers: followers,
        method: "api",
      });
    } catch (_apiErr) {
      port.postMessage({
        type: "info",
        message: "API unavailable — using page scroll. Keep this tab visible.",
      });
      try {
        var followers2 = await fetchFollowersViaDOM(pageUser, port);
        port.postMessage({
          type: "done",
          account: pageUser,
          followers: followers2,
          method: "dom",
        });
      } catch (domErr) {
        port.postMessage({
          type: "error",
          message: "Scan failed: " + domErr.message,
        });
      }
    }

    scanning = false;
  }

  // ── Port listener ────────────────────────────────────────────────────

  B.runtime.onConnect.addListener(function (port) {
    if (port.name !== "follower-scan") return;

    port.onMessage.addListener(function (msg) {
      if (msg.action === "scan") scanFollowers(port);
      else if (msg.action === "stop") stopRequested = true;
    });
  });
})();
