(function () {
  "use strict";

  var B = typeof browser !== "undefined" ? browser : chrome;
  var $ = function (id) { return document.getElementById(id); };

  var btnSave = $("btnSave");
  var btnCheck = $("btnCheck");
  var progressEl = $("progress");
  var resultsEl = $("results");
  var errorEl = $("error");
  var infoEl = $("info");
  var savedCountEl = $("savedCount");
  var lastScanEl = $("lastScan");
  var accountEl = $("account");

  var isScanning = false;

  // ── DOM helpers ────────────────────────────────────────────────────────

  function el(tag, attrs, children) {
    var node = document.createElement(tag);
    if (attrs) {
      Object.keys(attrs).forEach(function (k) {
        if (k === "className") node.className = attrs[k];
        else if (k === "style") node.style.cssText = attrs[k];
        else node.setAttribute(k, attrs[k]);
      });
    }
    if (typeof children === "string") {
      node.textContent = children;
    } else if (Array.isArray(children)) {
      children.forEach(function (c) { if (c) node.appendChild(c); });
    }
    return node;
  }

  function clearNode(node) {
    while (node.firstChild) node.removeChild(node.firstChild);
  }

  function buildUserList(usernames) {
    var ul = el("ul", { className: "unfollower-list" });
    usernames.forEach(function (u) {
      var a = el("a", { href: "https://x.com/" + u, target: "_blank" }, "@" + u);
      ul.appendChild(el("li", null, [a]));
    });
    return ul;
  }

  // ── UI helpers ─────────────────────────────────────────────────────────

  function showError(msg) {
    errorEl.textContent = msg;
    errorEl.classList.add("active");
  }

  function hideError() { errorEl.classList.remove("active"); }

  function showInfo(msg) {
    infoEl.textContent = msg;
    infoEl.classList.add("active");
  }

  function hideInfo() { infoEl.classList.remove("active"); }

  function showProgress(text) {
    progressEl.textContent = text;
    progressEl.classList.add("active");
  }

  function hideProgress() { progressEl.classList.remove("active"); }

  function setButtons(enabled) {
    btnSave.disabled = !enabled;
    btnCheck.disabled = !enabled;
    isScanning = !enabled;
  }

  function formatDate(ts) {
    if (!ts) return "never";
    var d = new Date(ts);
    return d.toLocaleDateString() + " " + d.toLocaleTimeString();
  }

  function loadStatus() {
    return B.storage.local.get(["followers", "scanDate", "account"]).then(function (data) {
      if (data.followers && data.followers.length > 0) {
        savedCountEl.textContent = data.followers.length.toLocaleString();
        lastScanEl.textContent = formatDate(data.scanDate);
        accountEl.textContent = data.account ? "@" + data.account : "\u2014";
      } else {
        savedCountEl.textContent = "none";
        lastScanEl.textContent = "never";
        accountEl.textContent = "\u2014";
      }
    });
  }

  // ── Scan runner ────────────────────────────────────────────────────────

  function runScan() {
    return new Promise(function (resolve, reject) {
      B.tabs.query({ active: true, currentWindow: true }).then(function (tabs) {
        var tab = tabs[0];

        if (
          !tab || !tab.url ||
          (tab.url.indexOf("x.com/") === -1 && tab.url.indexOf("twitter.com/") === -1)
        ) {
          reject(new Error("Open x.com/{username}/followers in the active tab."));
          return;
        }

        var port;
        try {
          port = B.tabs.connect(tab.id, { name: "follower-scan" });
        } catch (_e) {
          reject(new Error("Cannot connect to page. Reload x.com and try again."));
          return;
        }

        port.onMessage.addListener(function (msg) {
          if (msg.type === "error") {
            reject(new Error(msg.message));
            port.disconnect();
          } else if (msg.type === "started") {
            showProgress("Scanning @" + msg.account + "... 0 found");
          } else if (msg.type === "info") {
            showInfo(msg.message);
          } else if (msg.type === "progress") {
            var text = "Scanning... " + msg.count.toLocaleString() + " followers found";
            if (msg.waiting) text += " (rate limited, waiting...)";
            showProgress(text);
          } else if (msg.type === "done") {
            resolve({ account: msg.account, followers: msg.followers, method: msg.method });
            port.disconnect();
          }
        });

        port.onDisconnect.addListener(function () {
          if (isScanning) {
            reject(new Error("Connection lost. Reload x.com and try again."));
          }
        });

        port.postMessage({ action: "scan" });
      }).catch(function () {
        reject(new Error("Cannot access the current tab."));
      });
    });
  }

  // ── Save ───────────────────────────────────────────────────────────────

  btnSave.addEventListener("click", function () {
    if (isScanning) return;
    hideError();
    hideInfo();
    resultsEl.classList.remove("active");
    setButtons(false);

    runScan()
      .then(function (result) {
        return B.storage.local.set({
          account: result.account,
          followers: result.followers,
          scanDate: Date.now(),
        }).then(function () { return result; });
      })
      .then(function (result) {
        hideProgress();
        loadStatus();

        var methodLabel = result.method === "api" ? "via API" : "via page scroll";

        clearNode(resultsEl);
        resultsEl.appendChild(el("h3", null, "Saved"));
        resultsEl.appendChild(el("div", { className: "count ok" }, result.followers.length.toLocaleString()));
        resultsEl.appendChild(el("div", { style: "font-size:13px;color:#71767b" },
          "Followers saved as baseline for @" + result.account + " (" + methodLabel + ")."));
        resultsEl.classList.add("active");
      })
      .catch(function (e) {
        hideProgress();
        showError(e.message);
      })
      .finally(function () {
        setButtons(true);
      });
  });

  // ── Check ──────────────────────────────────────────────────────────────

  btnCheck.addEventListener("click", function () {
    if (isScanning) return;
    hideError();
    hideInfo();
    resultsEl.classList.remove("active");

    B.storage.local.get(["followers", "account"]).then(function (data) {
      if (!data.followers || data.followers.length === 0) {
        showError('No saved followers yet. Click "Save Current Followers" first.');
        return;
      }

      setButtons(false);

      runScan()
        .then(function (result) {
          hideProgress();

          var savedSet = new Set(data.followers.map(function (u) { return u.toLowerCase(); }));
          var currentSet = new Set(result.followers.map(function (u) { return u.toLowerCase(); }));

          var lost = data.followers.filter(function (u) { return !currentSet.has(u.toLowerCase()); });
          var gained = result.followers.filter(function (u) { return !savedSet.has(u.toLowerCase()); });

          clearNode(resultsEl);

          // Unfollowers section
          resultsEl.appendChild(el("h3", null, "Unfollowers"));
          if (lost.length === 0) {
            resultsEl.appendChild(el("div", { className: "count ok" }, "0"));
            resultsEl.appendChild(el("div", { style: "font-size:13px;color:#71767b" },
              "No one unfollowed you since last save."));
          } else {
            resultsEl.appendChild(el("div", { className: "count lost" }, String(lost.length)));
            resultsEl.appendChild(buildUserList(lost));
          }

          // New followers section
          if (gained.length > 0) {
            var section = el("div", { className: "new-followers" });
            section.appendChild(el("h3", null, "New Followers"));
            section.appendChild(el("div", { className: "count" }, String(gained.length)));
            section.appendChild(buildUserList(gained));
            resultsEl.appendChild(section);
          }

          // Summary
          resultsEl.appendChild(el("div", { style: "margin-top:10px;font-size:12px;color:#536471" },
            "Saved: " + data.followers.length.toLocaleString() +
            " \u2022 Current: " + result.followers.length.toLocaleString()));

          resultsEl.classList.add("active");
        })
        .catch(function (e) {
          hideProgress();
          showError(e.message);
        })
        .finally(function () {
          setButtons(true);
        });
    });
  });

  // ── Init ───────────────────────────────────────────────────────────────

  loadStatus();
})();
